<?php
session_start();
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método não permitido']);
    exit;
}

/* --- Validação de entrada --- */
$amount    = isset($_POST['amount']) ? (float)str_replace(',', '.', $_POST['amount']) : 0;
$cpf       = isset($_POST['cpf']) ? preg_replace('/\D/', '', $_POST['cpf']) : '';
$bonusCode = isset($_POST['bonus_code']) ? strtoupper(trim($_POST['bonus_code'])) : '';


if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Usuário não autenticado.']);
    exit;
}

/* --- Dependências --- */
require_once __DIR__ . '/../conexao.php';
require_once __DIR__ . '/../classes/zyropay.php';

try {
    $usuario_id = (int)$_SESSION['usuario_id'];

    /* --- Busca usuário (agora só temos usuario; nome/email podem não existir) --- */
    $stmt = $pdo->prepare("SELECT usuario, nome, email FROM usuarios WHERE id = :id LIMIT 1");
    $stmt->execute([':id' => $usuario_id]);
    $u = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$u) throw new Exception('Usuário não encontrado.');

    // Nome a ser enviado ao gateway (obrigatório em vários PSPs)
    $payerName = $u['nome'] ?: $u['usuario'] ?: 'Cliente Box';
    $payerName = mb_substr(trim($payerName), 0, 80);

    /* --- Garantir colunas opcionais na tabela depositos (ignora erro se não suportar IF NOT EXISTS) --- */
    try {
        $pdo->exec("ALTER TABLE depositos ADD COLUMN IF NOT EXISTS bonus_code VARCHAR(100) NULL");
        $pdo->exec("ALTER TABLE depositos ADD COLUMN IF NOT EXISTS bonus_percent DECIMAL(5,2) NULL");
        $pdo->exec("ALTER TABLE depositos ADD COLUMN IF NOT EXISTS bonus_amount DECIMAL(10,2) NOT NULL DEFAULT 0.00");
        $pdo->exec("ALTER TABLE depositos ADD COLUMN IF NOT EXISTS bonus_applied TINYINT(1) NOT NULL DEFAULT 0");
        $pdo->exec("ALTER TABLE depositos ADD COLUMN IF NOT EXISTS copiaecola TEXT NULL");
        $pdo->exec("ALTER TABLE depositos ADD COLUMN IF NOT EXISTS idempotency_key VARCHAR(191) NULL");
        $pdo->exec("ALTER TABLE depositos ADD COLUMN IF NOT EXISTS expires_at VARCHAR(64) NULL");
    } catch (Throwable $e) {
        // ok
    }

    /* --- Bônus (placeholder) --- */
    $bonus_percent = null;
    $bonus_amount  = 0.00;
    if ($bonusCode !== '') {
        // sua lógica de bônus aqui
    }

    /* --- Config da requisição ao PSP --- */
    $protocol    = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
    $host        = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $postbackUrl = $protocol . $host . '/callback/zyropay.php';

    $idempotencyKey = 'idemp_' . bin2hex(random_bytes(12));

  // Dados do pagador a enviar para o gateway
// Gera email fallback caso o usuário não tenha email cadastrado
$baseUser  = !empty($u['usuario']) ? preg_replace('/[^a-z0-9]/i', '', $u['usuario']) : ('user' . (int)$usuario_id);
$fallbackEmail = strtolower($baseUser) . '@box.com';

$customerEmail = !empty($u['email']) ? $u['email'] : $fallbackEmail;

$customerData = [
    'name'     => $payerName,
    'document' => (!empty($cpf) ? $cpf : '00000000191'), // fallback se não informar
    'email'    => $customerEmail
];

/* --- Criação do PIX via classe ZyroPay --- */
$result = ZyroPay::createPix(
    $pdo,
    $customerData,
    $amount,
    $postbackUrl,
    $idempotencyKey
);


    if (empty($result['transaction_id']) || empty($result['qr_code'])) {
        throw new Exception('Falha ao gerar o PIX. Resposta inesperada da ZyroPay.');
    }

    /* --- Persistência --- */
    @file_put_contents(
        __DIR__ . '/../payment.log',
        date('Y-m-d H:i:s') . " TX_CREATED: " . $result['transaction_id'] . PHP_EOL,
        FILE_APPEND
    );

    $stmt = $pdo->prepare("
        INSERT INTO depositos
        (transactionId, user_id, nome, cpf, valor, status, qrcode, copiaecola, gateway, idempotency_key, bonus_code, bonus_percent, bonus_amount, expires_at, created_at)
        VALUES
        (:transactionId, :user_id, :nome, :cpf, :valor, 'PENDING', :qrcode, :copiaecola, 'zyropay', :idempotency_key, :bonus_code, :bonus_percent, :bonus_amount, :expires_at, NOW())
    ");
    $stmt->execute([
        ':transactionId'   => $result['transaction_id'],
        ':user_id'         => $usuario_id,
        ':nome'            => $payerName,
        ':cpf'             => $cpf,
        ':valor'           => $amount,
        ':qrcode'          => $result['qr_code']        ?? null,
        ':copiaecola'      => $result['copy_paste_key'] ?? null,
        ':idempotency_key' => $idempotencyKey,
        ':bonus_code'      => ($bonusCode !== '' ? $bonusCode : null),
        ':bonus_percent'   => $bonus_percent,
        ':bonus_amount'    => $bonus_amount,
        ':expires_at'      => $result['expires_at']     ?? null,
    ]);

    $_SESSION['transactionId'] = $result['transaction_id'];

    echo json_encode([
        'gateway'       => 'zyropay',
        'transactionId' => $result['transaction_id'],
        'qrcode'        => $result['qr_code'],
        'copiaecola'    => $result['copy_paste_key'] ?? null,
        'expiresAt'     => $result['expires_at']     ?? null,
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
