<?php
session_start();
header('Content-Type: application/json');
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

// CONFIGURAÇÃO DE LOGS - ALTERE AQUI PARA ATIVAR/DESATIVAR
define('DEBUG_MODE', false); // true = logs ativos | false = logs desativados
define('LOG_FILE', 'logs.txt');

// Função para gravar logs apenas se DEBUG_MODE estiver ativo
function writeLog($message) {
    if (DEBUG_MODE) {
        file_put_contents(LOG_FILE, date('d/m/Y H:i:s') . " - " . $message . PHP_EOL, FILE_APPEND);
    }
}

// === UTMIFY: helper p/ enviar pedido ===
function utmify_send_order(array $payload, string $token) {
    $ch = curl_init("https://api.utmify.com.br/api-credentials/orders");
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Content-Type: application/json",
        "x-api-token: " . $token
    ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload, JSON_UNESCAPED_UNICODE));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    curl_close($ch);

    if ($err) writeLog("UTMIFY ERR: " . $err);
    writeLog("UTMIFY RESP: " . $resp);
}

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método não permitido']);
    exit;
}

$rawInput = file_get_contents('php://input');
$data = json_decode($rawInput, true);

writeLog("PAYLOAD PIXUP: " . print_r($data, true));
writeLog("----------------------------------------------------------");

if (!isset($data['requestBody'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Payload inválido']);
    exit;
}

$body          = $data['requestBody'];
$paymentType   = $body['paymentType']   ?? '';
$status        = $body['status']        ?? '';
$transactionId = $body['transactionId'] ?? '';

if ($paymentType !== 'PIX' || $status !== 'PAID' || empty($transactionId)) {
    http_response_code(400);
    echo json_encode(['error' => 'Dados insuficientes ou transação não paga']);
    exit;
}

require_once __DIR__ . '/../conexao.php';

try {
    $pdo->beginTransaction();
    
    writeLog("INICIANDO PROCESSO PARA TXN: " . $transactionId);
    
$stmt = $pdo->prepare("
  SELECT
    id, user_id, valor, status, created_at,
    utm_source, utm_medium, utm_campaign, utm_content, utm_term,
    src, sck, utm_json
  FROM depositos
  WHERE transactionId = :txid
  LIMIT 1 FOR UPDATE
");

    $stmt->execute([':txid' => $transactionId]);
    $deposito = $stmt->fetch();
    
    if (!$deposito) {
        $pdo->commit();
        writeLog("ERRO: Depósito não encontrado para TXN: " . $transactionId);
        http_response_code(404);
        echo json_encode(['error' => 'Depósito não encontrado']);
        exit;
    }
    
    writeLog("DEPÓSITO ENCONTRADO: " . print_r($deposito, true));
    
    if ($deposito['status'] === 'PAID') {
        $pdo->commit();
        echo json_encode(['message' => 'Este pagamento já foi aprovado']);
        exit;
    }
    
   // Verificar se é o primeiro depósito do usuário (antes de marcar como PAID!)
$stmt = $pdo->prepare("
    SELECT COUNT(*) AS total_pagos
    FROM depositos
    WHERE user_id = :uid
      AND UPPER(status) = 'PAID'
");
$stmt->execute([':uid' => $deposito['user_id']]);
$depositosAnteriores = (int)$stmt->fetchColumn();

writeLog("DEPÓSITOS ANTERIORES PAGOS (ANTES DE MARCAR COMO PAID): {$depositosAnteriores}");

// Atualiza o status para PAID (idempotente)
$stmt = $pdo->prepare("
    UPDATE depositos
       SET status = 'PAID',
           updated_at = NOW()
     WHERE id = :id
       AND UPPER(status) <> 'PAID'
");
$stmt->execute([':id' => $deposito['id']]);
writeLog("DEPÓSITO ATUALIZADO PARA PAID (linhas afetadas: {$stmt->rowCount()})");

// COMMIT imediato: libera o consult_pix ver PAID e o JS redirecionar
if ($pdo->inTransaction()) { $pdo->commit(); }

    
    // === UTMIFY: montar e enviar pedido "paid" ===

// pegar created_at do depósito e converter pra UTC
$createdAtUtc = (new DateTime(($deposito['created_at'] ?? 'now'), new DateTimeZone('America/Bahia')))
    ->setTimezone(new DateTimeZone('UTC'))
    ->format('Y-m-d H:i:s');

// approvedDate = agora em UTC
$approvedAtUtc = (new DateTime('now', new DateTimeZone('America/Bahia')))
    ->setTimezone(new DateTimeZone('UTC'))
    ->format('Y-m-d H:i:s');

// dados do usuário
$userRow = $pdo->prepare("SELECT nome, email, telefone, cpf FROM usuarios WHERE id = :uid LIMIT 1");
$userRow->execute([':uid' => $deposito['user_id']]);
$user = $userRow->fetch(PDO::FETCH_ASSOC) ?: [];

// valores
$valorDeposito      = (float)$deposito['valor'];
$valorEmCentavos    = (int) round($valorDeposito * 100);
$gatewayFeeInCents  = (int) round($valorEmCentavos * 0.04); // ajuste sua taxa real
$userCommissionInCents = max(0, $valorEmCentavos - $gatewayFeeInCents);

// UTM do depósito (webhook é server-to-server)
$tracking = [
    "src"          => $deposito['src']          ?? null,
    "sck"          => $deposito['sck']          ?? null,
    "utm_source"   => $deposito['utm_source']   ?? null,
    "utm_campaign" => $deposito['utm_campaign'] ?? null,
    "utm_medium"   => $deposito['utm_medium']   ?? null,
    "utm_content"  => $deposito['utm_content']  ?? null,
    "utm_term"     => $deposito['utm_term']     ?? null,
];

if (!empty($deposito['utm_json'])) {
    $j = json_decode($deposito['utm_json'], true) ?: [];
    // adiciona se existirem
    foreach (['gclid','fbclid','fbp','fbc'] as $k) {
        if (!empty($j[$k])) $tracking[$k] = $j[$k];
    }
}

// limpa nulos/vazios
$tracking = array_filter($tracking, fn($v) => !is_null($v) && $v !== '');

$tracking['event_id'] = 'oid-' . (string)$transactionId;


$payload = [
    "orderId"       => (string)$transactionId, // use seu ID único do pedido/tx
    "platform"      => "Raspa365",
    "paymentMethod" => "pix",
    "status"        => "paid",
    "createdAt"     => $createdAtUtc,
    "approvedDate"  => $approvedAtUtc,
    "refundedAt"    => null,
    "customer" => [
        "name"     => $user['nome']     ?? 'Cliente',
        "email"    => $user['email']    ?? 'sem-email@raspa365.com',
        "phone"    => $user['telefone'] ?? null,
        "document" => $user['cpf']      ?? null,
        "country"  => "BR",
        "ip"       => $_SERVER['REMOTE_ADDR'] ?? null,
    ],
    "products" => [[
        "id"           => "deposito-pix",
        "name"         => "Depósito PIX",
        "planId"       => null,
        "planName"     => null,
        "quantity"     => 1,
        "priceInCents" => $valorEmCentavos,
    ]],
    "trackingParameters" => $tracking,
    "commission" => [
        "totalPriceInCents"     => $valorEmCentavos,
        "gatewayFeeInCents"     => $gatewayFeeInCents,
        "userCommissionInCents" => $userCommissionInCents
        // "currency" => "BRL" // opcional
    ],
    "isTest" => false
];

// envie com seu token real da UTMify
$UTMIFY_TOKEN = "GrmgTEoMK9U38c3XtsqyHrtEc59J4ZpODygu";
utmify_send_order($payload, $UTMIFY_TOKEN);

// Marcar no banco que já enviamos pra UTMify (idempotência p/ polling)
$stmt = $pdo->prepare("
  UPDATE depositos
     SET utm_json = JSON_SET(COALESCE(utm_json, '{}'), '$.utmify_tracked', true)
   WHERE transactionId = :tx
   LIMIT 1
");
$stmt->execute([':tx' => $transactionId]);

    
    // Buscar config de bônus
    $configBonus = $pdo->query("SELECT multiplicador, rollover FROM bonus_config LIMIT 1")->fetch(PDO::FETCH_ASSOC);
    $multiplicador = isset($configBonus['multiplicador']) ? (float)$configBonus['multiplicador'] : 1;
    $rollover = isset($configBonus['rollover']) ? (int)$configBonus['rollover'] : 4;

    // Calcular valores - aplicar bônus apenas se for o primeiro depósito
    $valorDeposito = (float)$deposito['valor'];
    $valorBonus = ($depositosAnteriores == 0 && $multiplicador > 1) ? $valorDeposito * ($multiplicador - 1) : 0;
    $totalCreditar = $valorDeposito + $valorBonus;

    // Creditar saldo real e bônus separadamente
    $stmt = $pdo->prepare("
        UPDATE usuarios 
           SET saldo = saldo + :total,
               saldo_real = saldo_real + :real,
               saldo_bonus = saldo_bonus + :bonus
         WHERE id = :uid
    ");
    $stmt->execute([
        ':total' => $totalCreditar,
        ':real' => $valorDeposito,
        ':bonus' => $valorBonus,
        ':uid' => $deposito['user_id']
    ]);

    writeLog("✅ SALDO CREDITADO: real = R$ {$valorDeposito}, bônus = R$ {$valorBonus}, total = R$ {$totalCreditar} para usuário " . $deposito['user_id']);

    // Atualizar total_rollover com base no valor do depósito real (3x)
    $totalRolloverAdd = $valorDeposito * 3;

    $stmt = $pdo->prepare("
        UPDATE usuarios 
           SET total_rollover = total_rollover + :rollover 
         WHERE id = :uid
    ");
    $stmt->execute([
        ':rollover' => $totalRolloverAdd,
        ':uid' => $deposito['user_id']
    ]);

    writeLog("📈 ROLLOVER ATUALIZADO: +R$ {$totalRolloverAdd} para usuário {$deposito['user_id']}");

    // VERIFICAÇÃO PARA CPA (APENAS PRIMEIRO DEPÓSITO)
    $stmt = $pdo->prepare("SELECT indicacao FROM usuarios WHERE id = :uid");
    $stmt->execute([':uid' => $deposito['user_id']]);
    $usuario = $stmt->fetch();
    
    writeLog("USUÁRIO DATA: " . print_r($usuario, true));
    
    if ($usuario && !empty($usuario['indicacao'])) {
        writeLog("USUÁRIO TEM INDICAÇÃO: " . $usuario['indicacao']);
        
        // CPA só é pago se este for o PRIMEIRO depósito aprovado do usuário
        if ($depositosAnteriores == 0) {
            writeLog("É O PRIMEIRO DEPÓSITO, VERIFICANDO AFILIADO");
            
            $stmt = $pdo->prepare("SELECT id, comissao_cpa, banido FROM usuarios WHERE id = :afiliado_id");
            $stmt->execute([':afiliado_id' => $usuario['indicacao']]);
            $afiliado = $stmt->fetch();
            
            writeLog("AFILIADO DATA: " . print_r($afiliado, true));
            
            if ($afiliado && $afiliado['banido'] != 1 && !empty($afiliado['comissao_cpa'])) {
                $comissao = $afiliado['comissao_cpa'];
                
                // Credita a comissão CPA para o afiliado
                $stmt = $pdo->prepare("UPDATE usuarios SET saldo = saldo + :comissao WHERE id = :afiliado_id");
                $stmt->execute([
                    ':comissao' => $comissao,
                    ':afiliado_id' => $afiliado['id']
                ]);
                
                // Tenta inserir na tabela transacoes_afiliados
                try {
                    $stmt = $pdo->prepare("INSERT INTO transacoes_afiliados
                                          (afiliado_id, usuario_id, deposito_id, valor, created_at)
                                          VALUES (:afiliado_id, :usuario_id, :deposito_id, :valor, NOW())");
                    $stmt->execute([
                        ':afiliado_id' => $afiliado['id'],
                        ':usuario_id' => $deposito['user_id'],
                        ':deposito_id' => $deposito['id'],
                        ':valor' => $comissao
                    ]);
                } catch (Exception $insertError) {
                    writeLog("ERRO AO INSERIR TRANSAÇÃO AFILIADO: " . $insertError->getMessage());
                }
                
                writeLog("CPA PAGO: Afiliado {$afiliado['id']} recebeu R$ {$comissao} pelo primeiro depósito do usuário {$deposito['user_id']}");
            } else {
                writeLog("CPA NÃO PAGO: Afiliado inválido ou sem comissão");
            }
        } else {
            writeLog("CPA NÃO PAGO: Usuário {$deposito['user_id']} já teve {$depositosAnteriores['total_pagos']} depósito(s) aprovado(s) anteriormente");
        }
    } else {
        writeLog("USUÁRIO SEM INDICAÇÃO");
    }
    
    if ($pdo->inTransaction()) { $pdo->commit(); }
writeLog("TRANSAÇÃO FINALIZADA COM SUCESSO");
echo json_encode(['message' => 'OK']);
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) { $pdo->rollBack(); }
    writeLog("ERRO GERAL: " . $e->getMessage());
    writeLog("STACK TRACE: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode(['error' => 'Erro interno']);
    exit;
}