<?php
class Zyropay
{
    private const CONNECT_TIMEOUT = 10;
    private const TIMEOUT         = 20;
    private const USER_AGENT      = 'ZyropayClient/1.0';

    /* ====================== PONTO DE ENTRADA AUTO ====================== */
    public static function createPix(PDO $pdo, array $payer, float $amount, string $externalId, string $postbackUrl, array $options = [], ?string $idempotencyKey = null): array
    {
        $postbackUrl = "https://boxmilionario.site/callback/zyropay.php";
        self::assertPayer($payer);
        self::assertAmount($amount);
        self::assertUrl($postbackUrl);

        [$baseUrl, $ci, $cs] = self::getCredentials($pdo);

        // 1) Tenta V1 (ws.suitpay.app/api/v1/gateway/pix-payment)
       

        // 2) Fallback V2 (app.suitpay.app/v2...) com OAuth client_credentials
        return self::createPixV2OAuth($ci, $cs, $payer, $amount, $externalId, $postbackUrl, $options, $idempotencyKey);
    }

    /* ====================== IMPLEMENTAÇÃO V1 (SEM OAUTH) ====================== */
    public static function createPixV1(PDO $pdo, array $payer, float $amount, string $externalId, string $postbackUrl, array $options = [], ?string $idempotencyKey = null): array
    {
        [$baseUrl, $ci, $cs] = self::getCredentials($pdo);
        return self::createPixV1WithBase($baseUrl, $ci, $cs, $payer, $amount, $externalId, $postbackUrl, $options, $idempotencyKey);
    }

    private static function createPixV1WithBase(string $baseUrl, string $ci, string $cs, array $payer, float $amount, string $externalId, string $postbackUrl, array $options = [], ?string $idempotencyKey = null): array
    {
        
        $endpoint = $baseUrl . '/payment/pix/generate-pix';

        $payload = [
            'value'        => number_format($amount, 2, '.', ''),
            'externalId'   => $externalId,
            'expiration'   => 0,
        ];

        if (!empty($options['split'])) {
            $payload['split'] = array_values($options['split']);
        }

        $headers = [
            'Content-Type: application/json',
            'Accept: application/json',
            'ci: ' . $ci,
            'cs: ' . $cs,
            'User-Agent: ' . self::USER_AGENT,
            'Expect:'
        ];
        if ($idempotencyKey) $headers[] = 'Idempotency-Key: ' . $idempotencyKey;

        $resp = self::curlJsonPost($endpoint, $payload, $headers);
        $http = $resp['http_code'];
        $body = $resp['body'];
        $json = json_decode($body, true);
        
        if ($resp['curl_error']) {
            self::log('V1_CURL', $resp['curl_error'] . " (HTTP {$http})");
            throw new \Exception('Erro SuitPay v1: ' . $resp['curl_error'] . ' (HTTP ' . $http . ')');
        }
        if ($http < 200 || $http >= 300 || !is_array($json)) {
            self::log('V1_HTTP', "HTTP {$http} - " . $body);
            throw new \Exception('Falha ao gerar QR Code (v1). HTTP ' . $http . ' - ' . (is_array($json) ? ($json['message'] ?? 'No message') : 'No message'));
        }

        $transactionId = $json['paymentId'] ?? ($json['paymentId'] ?? $externalId);
        $qrcode        = $json['qrcode'] ?? ($json['qrCode'] ?? null);
        $copyPaste     = $json['copyPaste'] ?? ($json['copy_and_paste'] ?? $json['emvqrcps'] ?? null);
        $expiresAt     = $json['expiresAt'] ?? ($json['expirationDate'] ?? null);

        if (!$transactionId || !$qrcode) {
            self::log('V1_PARSE', $body);
            throw new \Exception('Resposta inesperada da SuitPay (v1).');
        }

        return [
            'transaction_id' => $transactionId,
            'qrcode'        => $qrcode,
            'copyPaste'     => $copyPaste,
            'expiresAt'     => $expiresAt,
            'raw'           => $json,
        ];
    }

    /* ====================== IMPLEMENTAÇÃO V2 (OAUTH) ====================== */
    private static function createPixV2OAuth(string $ci, string $cs, array $payer, float $amount, string $externalId, string $postbackUrl, array $options = [], ?string $idempotencyKey = null): array
    {
        $authUrl = 'https://gateway-zyropay-api.rancher.codefabrik.dev/cli/client/authenticate';

        $payload =[
            'clientId' => '68a37617be51bef54951defa',
            'password' => 'hkU3kpsVTv2L',
        ];
        
        $t = self::curlJsonPost(
            $authUrl,
            $payload, 
            [
                'accept: */*',
                'Content-Type: application/json',
            ]
        );
        if ($t['curl_error']) {
            self::log('V2_TOKEN_CURL', $t['curl_error'] . " (HTTP {$t['http_code']})");
            throw new \Exception('Falha na autenticação SuitPay (v2): ' . $t['curl_error'] . ' (HTTP ' . $t['http_code'] . ')');
        }
        $tok = json_decode($t['body'], true);
        
     
        if (!is_array($tok) || empty($tok['data']['token'])) {
            self::log('V2_TOKEN_HTTP', "HTTP {$t['http_code']} - " . $t['body']);
            throw new \Exception('Access token inválido (v2). HTTP ' . $t['http_code'] );
        }
        $accessToken = $tok['data']['token'];
        

        $endpoint = 'https://gateway-zyropay-api.rancher.codefabrik.dev/cli/payment/pix/generate-pix';
        $payload = [
            'value'        => number_format($amount, 2, '.', ''),
            'externalId'   => $externalId,
            'expiration'   => 0,
        ];
        if (!empty($options['split'])) {
            $payload['split'] = array_values($options['split']);
        }

        $headers = [
            'Authorization: Bearer ' . $accessToken,
            'Content-Type: application/json',
            'Accept: application/json',
            'accept: */*'
        ];
        if ($idempotencyKey) $headers[] = 'Idempotency-Key: ' . $idempotencyKey;

        $r = self::curlJsonPost($endpoint, $payload, $headers);
        $http = $r['http_code'];
        $body = $r['body'];
        $json = json_decode($body, true);

        if ($r['curl_error']) {
            self::log('V2_CURL', $r['curl_error'] . " (HTTP {$http})");
            throw new \Exception('Erro SuitPay v2: ' . $r['curl_error'] . ' (HTTP ' . $http . ')');
        }
        if ($http < 200 || $http >= 300 || !is_array($json)) {
            self::log('V2_HTTP', "HTTP {$http} - " . $body);
            throw new \Exception('Falha ao gerar QR Code (v2). HTTP ' . $http . ' - ' . (is_array($json) ? ($json['message'] ?? 'No message') : 'No message'));
        }

        $transactionId = $json['data']['paymentId'] ?? ($json['data']['paymentId'] ?? $externalId);
        $qrcode        = $json['data']['pix'] ?? ($json['data']['pix'] ?? null);
        $copyPaste     = $json['data']['pix'] ?? ($json['data']['pix'] ?? $json['data']['emvqrcps'] ?? null);
        $expiresAt     = $json['data']['expiresAt'] ?? ($json['data']['expirationDate'] ?? null);
        
        self::log('tid', $transactionId);
        if (!$transactionId || !$qrcode) {
            self::log('V2_PARSE', $body);
            throw new \Exception('asd' . json_encode($json['data']['paymentId']));
        }

        return [
            'transaction_id' => $transactionId,
            'qr_code'        => $qrcode,
            'copy_paste_key'     => $copyPaste,
            'expires_at'     => $expiresAt,
            'raw'           => $json,
        ];
    }

    /* ====================== Helpers HTTP ====================== */
    private static function curlJsonPost(string $url, array $payload, array $headers): array
    {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
            CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
            CURLOPT_CONNECTTIMEOUT => self::CONNECT_TIMEOUT,
            CURLOPT_TIMEOUT        => self::TIMEOUT,
        ]);
        $body      = curl_exec($ch);
        $curlError = curl_error($ch);
        $httpCode  = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return ['body'=>$body, 'curl_error'=>$curlError, 'http_code'=>$httpCode];
    }

    private static function curlPostForm(string $url, array $form, array $headers): array
    {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => http_build_query($form),
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
            CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
            CURLOPT_CONNECTTIMEOUT => self::CONNECT_TIMEOUT,
            CURLOPT_TIMEOUT        => self::TIMEOUT,
        ]);
        $body      = curl_exec($ch);
        $curlError = curl_error($ch);
        $httpCode  = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return ['body'=>$body, 'curl_error'=>$curlError, 'http_code'=>$httpCode];
    }

    /* ====================== Validadores & Utils ====================== */
    private static function getCredentials(PDO $pdo): array
    {
     
        $baseUrl = 'https://gateway-zyropay-api.rancher.codefabrik.dev/cli';
        $ci      = 'a';
        $cs      = 'b';
        if ($baseUrl === '' || $ci === '' || $cs === '') throw new \Exception('Campos url/ci/cs inválidos.');
        return [rtrim($baseUrl, '/'), $ci, $cs];
    }

    private static function assertPayer(array $payer): void
    {
        foreach (['name','document','email'] as $k) {
            if (empty($payer[$k])) throw new \Exception("Payer.$k ausente.");
        }
        $doc = preg_replace('/\D/','',(string)$payer['document']);
        if (strlen($doc) !== 11) throw new \Exception('CPF inválido.');
    }

    private static function assertAmount(float $amount): void
    {
        if ($amount <= 0) throw new \Exception('Valor inválido.');
    }

    private static function assertUrl(string $url): void
    {
        if (!preg_match('~^https?://~i',$url)) throw new \Exception('post
        $postbackUrl =>"https://boxmilionario.site/backUrl inválida.');
    }

    private static function log(string $label, $data): void
    {
        try {
            @file_put_contents(__DIR__ . '/../runtime_suitpay.log',
                date('Y-m-d H:i:s') . " [$label] " . (is_string($data)?$data:json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES)) . "\n",
                FILE_APPEND
            );
        } catch (\Throwable $e) {}
    }
}
