<?php

class Zyropay
{
    private const CONNECT_TIMEOUT = 10;
    private const TIMEOUT         = 20;
    private const USER_AGENT      = 'ZyropayClient/1.0';

    /**
     * Ponto de entrada para criar uma cobrança PIX com Zyropay.
     *
     * @param PDO $pdo Conexão com o banco de dados.
     * @param array $payer Dados do pagador ['name', 'document', 'email'].
     * @param float $amount Valor da cobrança.
     * @param string $externalId ID externo para sua referência.
     * @param string $postbackUrl URL para receber notificações de pagamento.
     * @param array $options Opções adicionais (se houver).
     * @param string|null $idempotencyKey Chave de idempotência para evitar cobranças duplicadas.
     * @return array Retorna os dados do PIX gerado.
     * @throws \Exception Em caso de erro.
     */
    public static function createPix(PDO $pdo, array $payer, float $amount, string $externalId, string $postbackUrl, array $options = [], ?string $idempotencyKey = null): array
    {
        throw "asd";
        // Valida os dados de entrada
        self::assertPayer($payer);
        self::assertAmount($amount);
        self::assertUrl($postbackUrl);

        // Obtém as credenciais do banco de dados
        [$apiUrl, $apiKey] = self::getCredentials($pdo);

        // ... ZYROPAY DOCS ...
        // Confirme o endpoint correto na documentação da Zyropay.
        $endpoint = $apiUrl . '/v1/payments/pix'; // Exemplo, ajuste conforme a documentação

        // Monta o corpo da requisição (payload)
        $payload = [
            // ... ZYROPAY DOCS ...
            // Abaixo está uma estrutura de exemplo.
            // Você DEVE ajustar os nomes dos campos conforme a documentação da Zyropay.
            'transaction_amount' => number_format($amount, 2, '.', ''),
            'description'        => 'Pagamento para ' . $payer['name'],
            'external_reference' => $externalId,
            'notification_url'   => $postbackUrl,
            'payer' => [
                'first_name'   => $payer['name'],
                'email'        => $payer['email'],
                'identification' => [
                    'type'   => 'CPF',
                    'number' => preg_replace('/\D/', '', $payer['document']),
                ],
            ],
        ];

        // Monta os cabeçalhos (headers)
        $headers = [
            'Content-Type: application/json',
            'Accept: application/json',
            // ... ZYROPAY DOCS ...
            // Confirme o método de autenticação. Pode ser 'Authorization: Bearer ' . $apiKey
            // ou uma chave customizada como 'X-API-Key: ' . $apiKey
            'Authorization: Bearer ' . $apiKey,
            'User-Agent: ' . self::USER_AGENT,
            'Expect:',
        ];

        if ($idempotencyKey) {
            // ... ZYROPAY DOCS ...
            // Confirme o nome do header de idempotência, se suportado.
            $headers[] = 'Idempotency-Key: ' . $idempotencyKey;
        }

        // Envia a requisição para a API
        $resp = self::curlJsonPost($endpoint, $payload, $headers);
        $http = $resp['http_code'];
        $body = $resp['body'];
        $json = json_decode($body, true);

        // Tratamento de erros
        if ($resp['curl_error']) {
            self::log('CURL_ERROR', $resp['curl_error'] . " (HTTP {$http})");
            throw new \Exception('Erro de comunicação com a Zyropay: ' . $resp['curl_error'] . ' (HTTP ' . $http . ')');
        }
        if ($http < 200 || $http >= 300 || !is_array($json)) {
            self::log('HTTP_ERROR', "HTTP {$http} - " . $body);
            throw new \Exception('Falha ao gerar QR Code na Zyropay. HTTP ' . $http . ' - ' . ($json['message'] ?? 'Erro desconhecido'));
        }

        // ... ZYROPAY DOCS ...
        // Ajuste os campos abaixo para corresponder à resposta da API da Zyropay.
        $transactionId = $json['id'] ?? $externalId;
        $copyPaste     = $json['point_of_interaction']['transaction_data']['qr_code'] ?? null;
        $qrcode        = $json['point_of_interaction']['transaction_data']['qr_code_base64'] ?? null;
        $expiresAt     = $json['date_of_expiration'] ?? null;

        if (!$transactionId || !$copyPaste || !$qrcode) {
            self::log('PARSE_ERROR', $body);
            throw new \Exception('Resposta inesperada da Zyropay.');
        }

        // Retorna os dados formatados
        return [
            'transactionId' => $transactionId,
            'qrcode'        => 'data:image/png;base64,' . $qrcode, // Assumindo que a resposta é base64
            'copyPaste'     => $copyPaste,
            'expiresAt'     => $expiresAt,
            'raw'           => $json,
        ];
    }

    /* ====================== Helpers HTTP ====================== */
    private static function curlJsonPost(string $url, array $payload, array $headers): array
    {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
            CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
            CURLOPT_CONNECTTIMEOUT => self::CONNECT_TIMEOUT,
            CURLOPT_TIMEOUT        => self::TIMEOUT,
        ]);
        $body      = curl_exec($ch);
        $curlError = curl_error($ch);
        $httpCode  = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return ['body' => $body, 'curl_error' => $curlError, 'http_code' => $httpCode];
    }

    /* ====================== Validadores & Utils ====================== */
    private static function getCredentials(PDO $pdo): array
    {
        // Altere a query para buscar as credenciais da Zyropay na sua tabela
        $stmt = $pdo->query("SELECT api_url, api_key FROM zyropay LIMIT 1");
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);

        if (!$row) {
            throw new \Exception('Credenciais Zyropay ausentes no banco de dados.');
        }
        $apiUrl = trim($row['api_url'] ?? '');
        $apiKey = trim($row['api_key'] ?? '');

        if ($apiUrl === '' || $apiKey === '') {
            throw new \Exception('Campos api_url/api_key da Zyropay inválidos.');
        }
        return [rtrim($apiUrl, '/'), $apiKey];
    }

    private static function assertPayer(array $payer): void
    {
        foreach (['name', 'document', 'email'] as $k) {
            if (empty($payer[$k])) {
                throw new \Exception("O campo Payer.'$k' é obrigatório.");
            }
        }
        $doc = preg_replace('/\D/', '', (string)$payer['document']);
        if (strlen($doc) !== 11) {
            throw new \Exception('CPF do pagador é inválido.');
        }
    }

    private static function assertAmount(float $amount): void
    {
        if ($amount <= 0) {
            throw new \Exception('O valor da cobrança deve ser maior que zero.');
        }
    }

    private static function assertUrl(string $url): void
    {
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            throw new \Exception('A postbackUrl fornecida é inválida.');
        }
    }

    private static function log(string $label, $data): void
    {
        try {
            $logMessage = date('Y-m-d H:i:s') . " [$label] " . (is_string($data) ? $data : json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)) . "\n";
            // Altere o caminho do arquivo de log se necessário
            @file_put_contents(__DIR__ . '/../runtime_zyropay.log', $logMessage, FILE_APPEND);
        } catch (\Throwable $e) {
            // Silencia erros de escrita de log
        }
    }
}
