<?php
session_start();
require 'db.php';

// Função para registrar logs em arquivo
function logStatus($message) {
    $log_file = 'log_status_transacao.txt';
    $date = date('[Y-m-d H:i:s]');
    $session_id = session_id();
    file_put_contents($log_file, "$date [$session_id] $message" . PHP_EOL, FILE_APPEND);
}

// Garante que a saída será sempre JSON
header('Content-Type: application/json; charset=utf-8');

try {
    // Verifica se o usuário está logado
    if (!isset($_SESSION['usuario_id'])) {
        logStatus("Erro: Usuário não autenticado");
        http_response_code(403);
        echo json_encode(['error' => 'Não autorizado']);
        exit;
    }

    $external_id = $_GET['external_id'] ?? '';
    $usuario_id = $_SESSION['usuario_id'];
    
    logStatus("Verificando transação: external_id=$external_id, usuario_id=$usuario_id");

    if (!$external_id) {
        logStatus("Erro: external_id ausente");
        http_response_code(400);
        echo json_encode(['error' => 'external_id ausente']);
        exit;
    }

    // Consulta o status da transação
    try {
        $stmt = $pdo->prepare("SELECT id, status, valor FROM transacoes_pix WHERE external_id = ? AND usuario_id = ?");
        $stmt->execute([$external_id, $usuario_id]);
        $transacao = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$transacao) {
            logStatus("Erro: Transação não encontrada com external_id=$external_id, usuario_id=$usuario_id");
            http_response_code(404);
            echo json_encode(['error' => 'Transação não encontrada']);
            exit;
        }
        
        logStatus("Transação encontrada: id={$transacao['id']}, status={$transacao['status']}");
        $status = $transacao['status'];
    } catch (PDOException $e) {
        logStatus("Erro de banco ao consultar transação: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao consultar status da transação']);
        exit;
    }

    // Consulta o saldo do usuário
    try {
        $stmt = $pdo->prepare("SELECT saldo FROM usuarios WHERE id = ?");
        $stmt->execute([$usuario_id]);
        $saldo = floatval($stmt->fetchColumn());
        logStatus("Saldo do usuário: $saldo");
    } catch (PDOException $e) {
        logStatus("Erro de banco ao consultar saldo: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao consultar saldo']);
        exit;
    }

    // Formata o resultado e retorna
    $resultado = [
        'status' => $status,
        'saldo' => number_format($saldo, 2, ',', '.')
    ];
    
    logStatus("Resposta: " . json_encode($resultado));
    echo json_encode($resultado);
    
} catch (Exception $e) {
    // Captura qualquer outro erro não tratado
    logStatus("Erro não tratado: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Erro interno no servidor']);
}
