<?php
require 'db.php';
header('Content-Type: application/json; charset=utf-8');

// ============ Função de Log ============
function logWebhook($message, $nivel = "INFO") {
    $log_file = "log_webhook_bspay.txt";
    $date = date('[Y-m-d H:i:s]');
    $prefix = "$date [$nivel] ";
    file_put_contents($log_file, $prefix . $message . PHP_EOL, FILE_APPEND);
}

// ============ Log básico da requisição ============
logWebhook("Método: " . $_SERVER['REQUEST_METHOD']);
logWebhook("Headers: " . print_r(getallheaders(), true));

// Leia o corpo cru ANTES de decodificar (preserva para assinatura, se usar)
$raw = file_get_contents('php://input');
logWebhook("Recebido (php://input): " . $raw);

$data = json_decode($raw, true);

// Fallback para $_POST se necessário
if (!$data && !empty($_POST)) {
    $data = $_POST;
    logWebhook("Fallback \$_POST: " . print_r($_POST, true));
}

if (!$data) {
    logWebhook("Erro: Dados inválidos ou vazios", "ERRO");
    http_response_code(400);
    echo json_encode(['error' => 'Dados inválidos ou vazios']);
    exit;
}

// A BSPay pode enviar dentro de "requestBody" ou no topo; suporte aos dois
$request = $data['requestBody'] ?? $data ?? null;

if (
    !$request
    || !isset($request['transactionId'])
    || !isset($request['external_id'])
    || !isset($request['status'])
) {
    logWebhook("Erro: Dados incompletos: " . json_encode($request), "ERRO");
    http_response_code(400);
    echo json_encode(['error' => 'Dados incompletos']);
    exit;
}

$transaction_id      = trim($request['transactionId']);
$external_id_webhook = trim($request['external_id']);
$status_webhook      = strtoupper(trim($request['status']));
$valor_pago          = floatval($request['amount'] ?? 0);
$data_aprovado       = $request['dateApproval'] ?? date('Y-m-d H:i:s');

$STATUS_OK = ['PAID','APPROVED','CONFIRMED','SUCCEEDED'];

if (in_array($status_webhook, $STATUS_OK, true)) {
    logWebhook("Status aprovado recebido: $status_webhook para external_id: $external_id_webhook");

    // Busca transação pendente
    try {
        $stmt = $pdo->prepare("SELECT * FROM transacoes_pix WHERE external_id = ? AND LOWER(status) = 'pendente' LIMIT 1");
        $stmt->execute([$external_id_webhook]);
        $transacao = $stmt->fetch();
    } catch (PDOException $e) {
        logWebhook("Erro ao buscar transação no banco: " . $e->getMessage(), "ERRO");
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao processar transação']);
        exit;
    }

    if ($transacao) {
        logWebhook("Transação pendente encontrada: ID {$transacao['id']}, usuario_id: {$transacao['usuario_id']}");

        try {
            // Atualiza status para aprovado
            $stmtUp = $pdo->prepare("UPDATE transacoes_pix SET status = 'aprovado', valor = ?, transaction_id = ? WHERE id = ?");
            $stmtUp->execute([$valor_pago, $transaction_id, $transacao['id']]);
            logWebhook("Status da transação atualizado para 'aprovado': ID {$transacao['id']}");
        } catch (PDOException $e) {
            logWebhook("Erro ao atualizar status da transação: " . $e->getMessage(), "ERRO");
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao atualizar transação']);
            exit;
        }

        // Busca configurações
        $config = $pdo->query("SELECT bonus_deposito, valor_comissao, valor_comissao_n2, rollover_multiplicador FROM configuracoes LIMIT 1")->fetch();
        $bonusPercent          = floatval($config['bonus_deposito']) / 100;
        $percentualComissaoN1  = floatval($config['valor_comissao']) / 100;
        $percentualComissaoN2  = floatval($config['valor_comissao_n2']) / 100;
        $rolloverMultiplicador = floatval($config['rollover_multiplicador']) ?: 2;

        $bonusValor = $valor_pago * $bonusPercent;
        $valorFinal = $valor_pago + $bonusValor;

        try {
            // Atualiza saldo
            $stmtSaldo = $pdo->prepare("UPDATE usuarios SET saldo = saldo + ? WHERE id = ?");
            $stmtSaldo->execute([$valorFinal, $transacao['usuario_id']]);
            logWebhook("Saldo atualizado: +$valorFinal para usuário {$transacao['usuario_id']} (Depósito: $valor_pago, Bônus: $bonusValor)");
        } catch (PDOException $e) {
            logWebhook("Erro ao atualizar saldo do usuário: " . $e->getMessage(), "ERRO");
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao atualizar saldo']);
            exit;
        }

        // Cria rollover se não existir
        $stmtRol = $pdo->prepare("SELECT COUNT(*) FROM rollover WHERE usuario_id = ? AND valor_deposito = ? AND finalizado = 0");
        $stmtRol->execute([$transacao['usuario_id'], $valor_pago]);
        if (!$stmtRol->fetchColumn()) {
            $valorRollover = $valor_pago * $rolloverMultiplicador;
            $stmtIR = $pdo->prepare("INSERT INTO rollover (usuario_id, valor_deposito, valor_necessario, valor_acumulado, finalizado, criado_em) VALUES (?, ?, ?, 0, 0, NOW())");
            $stmtIR->execute([$transacao['usuario_id'], $valor_pago, $valorRollover]);
        }

        // ======= Comissão Nível 1 =======
        $stmtInd = $pdo->prepare("SELECT indicado_por FROM usuarios WHERE id = ?");
        $stmtInd->execute([$transacao['usuario_id']]);
        $indicadorNivel1 = $stmtInd->fetchColumn();

        if ($indicadorNivel1) {
            $valorComissaoN1 = $valor_pago * $percentualComissaoN1;
            $stmtCI1 = $pdo->prepare("UPDATE usuarios SET comissao = comissao + ? WHERE id = ?");
            $stmtCI1->execute([$valorComissaoN1, $indicadorNivel1]);

            $stmtIC1 = $pdo->prepare("INSERT INTO comissoes (usuario_id, indicado_id, transacao_id, valor, nivel) VALUES (?, ?, ?, ?, 1)");
            $stmtIC1->execute([$indicadorNivel1, $transacao['usuario_id'], $transacao['id'], $valorComissaoN1]);

            // ======= Comissão Nível 2 =======
            $stmtInd2 = $pdo->prepare("SELECT indicado_por FROM usuarios WHERE id = ?");
            $stmtInd2->execute([$indicadorNivel1]);
            $indicadorNivel2 = $stmtInd2->fetchColumn();

            if ($indicadorNivel2) {
                $valorComissaoN2 = $valor_pago * $percentualComissaoN2;
                $stmtCI2 = $pdo->prepare("UPDATE usuarios SET comissao = comissao + ? WHERE id = ?");
                $stmtCI2->execute([$valorComissaoN2, $indicadorNivel2]);

                $stmtIC2 = $pdo->prepare("INSERT INTO comissoes (usuario_id, indicado_id, transacao_id, valor, nivel) VALUES (?, ?, ?, ?, 2)");
                $stmtIC2->execute([$indicadorNivel2, $transacao['usuario_id'], $transacao['id'], $valorComissaoN2]);
            }
        }

        logWebhook("Processamento completo com sucesso para transação ID {$transacao['id']}");
        http_response_code(200);
        echo json_encode(['message' => 'Depósito aprovado e processado']);
        exit;

    } else {
        logWebhook("Transação não encontrada ou já processada para external_id: $external_id_webhook", "AVISO");
        
        // Verifica se já foi processada (status=aprovado)
        try {
            $stmt = $pdo->prepare("SELECT id, status FROM transacoes_pix WHERE external_id = ? LIMIT 1");
            $stmt->execute([$external_id_webhook]);
            $transacao_existente = $stmt->fetch();
            
            if ($transacao_existente) {
                logWebhook("Transação já existe com status '{$transacao_existente['status']}': ID {$transacao_existente['id']}");
            } else {
                logWebhook("Nenhuma transação encontrada com external_id: $external_id_webhook", "AVISO");
            }
        } catch (PDOException $e) {
            logWebhook("Erro ao verificar transação existente: " . $e->getMessage(), "ERRO");
        }
        
        http_response_code(200);
        echo json_encode(['message' => 'Transação não encontrada ou já processada']);
        exit;
    }
}

// Outros status
logWebhook("Status ignorado: $status_webhook para external_id: $external_id_webhook");
http_response_code(200);
echo json_encode(['message' => "Status {$status_webhook} ignorado"]);
exit;
